<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;

class AdminController extends Controller
{
    /**
     * Get all users (Admin only)
     */
    public function getAllUsers(Request $request)
    {
        $users = User::orderBy('created_at', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $users->map(function ($user) {
                return [
                    'id' => $user->id,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'is_approved' => $user->is_approved,
                    'created_at' => $user->created_at,
                ];
            }),
        ]);
    }

    /**
     * Get pending users (Admin only)
     */
    public function getPendingUsers()
    {
        $pendingUsers = User::where('is_approved', false)
            ->where('role', 'user')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $pendingUsers->map(function ($user) {
                return [
                    'id' => $user->id,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'role' => $user->role,
                    'is_approved' => $user->is_approved,
                    'created_at' => $user->created_at,
                ];
            }),
        ]);
    }

    /**
     * Approve user (Admin only)
     */
    public function approveUser(Request $request, $id)
    {
        $user = User::findOrFail($id);

        if ($user->role === 'admin') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot modify admin users',
            ], 403);
        }

        $user->is_approved = true;
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'User approved successfully',
            'data' => [
                'id' => $user->id,
                'email' => $user->email,
                'phone' => $user->phone,
                'role' => $user->role,
                'is_approved' => $user->is_approved,
            ],
        ]);
    }

    /**
     * Reject/Disapprove user (Admin only)
     */
    public function rejectUser(Request $request, $id)
    {
        $user = User::findOrFail($id);

        if ($user->role === 'admin') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot modify admin users',
            ], 403);
        }

        $user->is_approved = false;
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'User rejected successfully',
            'data' => [
                'id' => $user->id,
                'email' => $user->email,
                'phone' => $user->phone,
                'role' => $user->role,
                'is_approved' => $user->is_approved,
            ],
        ]);
    }

    /**
     * Delete user (Admin only)
     */
    public function deleteUser($id)
    {
        $user = User::findOrFail($id);

        if ($user->role === 'admin') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete admin users',
            ], 403);
        }

        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully',
        ]);
    }
}
